//
// $Id: fmcrtc.h,v 1.5 2001/07/18 17:43:31 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//


#define BX_MAX_XSIZE 768
#define BX_MAX_YSIZE 512
#define X_TILESIZE 16
#define Y_TILESIZE 16
#define BX_NUM_X_TILES ((BX_MAX_XSIZE+X_TILESIZE-1)/X_TILESIZE)
#define BX_NUM_Y_TILES ((BX_MAX_YSIZE+Y_TILESIZE-1)/Y_TILESIZE)

#define DISP_16COLOR_0  1
#define DISP_16COLOR_1  2
#define DISP_256COLOR_0 4
#define DISP_256COLOR_1 8
#define DISP_HIGHCOLOR  16
#define DISP_TRUECOLOR  32

#define PEL_16_0  0
#define PEL_16_1  1
#define PEL_256_0 2
#define PEL_256_1 3

#define PEL_OFFSET_MASK   0xff000000
#define OFFSET_16COLOR_0  0x00000000
#define OFFSET_16COLOR_1  0x01000000
#define OFFSET_256COLOR_0 0x10000000
#define OFFSET_256COLOR_1 0x11000000
#define OFFSET_HIGHCOLOR  0x20000000
#define OFFSET_TRUECOLOR  0x30000000

typedef struct {
  Bit8u red;                /* 0xfd94 */
  Bit8u green;              /* 0xfd96 */
  Bit8u blue;               /* 0xfd92 */
} peldata;

#if BX_USE_VGA_SMF
#  define BX_VGA_SMF  static
#  define BX_VGA_THIS bx_vga.
#else
#  define BX_VGA_SMF
#  define BX_VGA_THIS this->
#endif

struct layer_info {
  unsigned depth;             /* 0 is don't display */
    signed top;
    signed bottom;
    signed left;
    signed right;
  unsigned xzoom2;
  unsigned yzoom;
    Bit32u layeraddr;
    Bit32u layermask;
    signed displeft;
    Bit32u lineoff;
    Bit32u dispoffh;
    Bit32u dispoffl;
    Bit32u vrollmask;
    Bit32u paloffset;
};
/*
       spherical scroll
       vrollmask != 0xfffffff
       dispoffl   = dispoff & vrollmask
       dispoffh   = dispoff & ~vrollmask

       cylindrical scroll
       vrollmask  = 0xffffffff
       dispoffl   = 0
       dispoffh   = dispoff

       addr = layeraddr+
               ((dispoffh+
	        ((dispoffl+(x-displeft)*2/xzoom*depth/8) & vrollmask)+
	         (y-top)/yzoom*lineoff)&layermask)

       y    = ((addr-layeraddr-dispoffh)&layermask)/lineoff*yzoom+top
       x    = ((addr-layeraddr-dispoffh-(y-top)/yzoom*lineoff
                -dispoffl)&vrollmask)*8/depth*xzoom/2+displeft
    */

#define	BORDERWIDTH	10
#define	DEFWIDTH15KHZ	640+BORDERWIDTH*2
#define	DEFHEIGHT15KHZ	480+BORDERWIDTH*2
#define	DEFWIDTH24KHZ	640+BORDERWIDTH*2
#define	DEFHEIGHT24KHZ	480+BORDERWIDTH*2
#define	DEFWIDTH31KHZ	640+BORDERWIDTH*2
#define	DEFHEIGHT31KHZ	480+BORDERWIDTH*2

class bx_vga_c : public logfunctions {
public:

  bx_vga_c(void);
  ~bx_vga_c(void);
  BX_VGA_SMF void   init(bx_devices_c *, bx_cmos_c *cmos);
  BX_VGA_SMF Bit8u  mem_read(Bit32u addr);
  // Note: either leave value of type Bit8u, or mask it when
  //       used to 8 bits, in memory.cc
  BX_VGA_SMF void   mem_write(Bit32u addr, Bit8u value);
  BX_VGA_SMF void   redraw_area(unsigned x0, unsigned y0,
                                unsigned width, unsigned height);

private:

	int          	vsync_timer_handle;
  static Bit32u read_handler(void *this_ptr, Bit32u address, unsigned io_len);
  static void   write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len);
  static void   write_handler_no_log(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len);

  struct {
    struct {
      Bit8u addr;                 /* 0x0450 */
      Bit8u reg[8];               /* 0x0452 */
      Bit8u mem[0x20000];         /* 128KB */
      } sprite;

    struct {
      Bit8u shifter_addr;         /* 0x0448 */
      Bit8u shifter[2];           /* 0x044a */
      Bit8u output_ctrl;          /* 0xfda0 */
      } misc_output;

    struct {
      Bit8u addr;                 /* 0x0440 */
      Bit16u reg[32];             /* 0x0442 */
      } CRTC;

    struct {
      Boolean DPMD;               /* 0x044c bit 7 */
      Bit8u degipal[8];           /* 0xfd98 -- 0xfd9f */
      Bit8u disp_page_select;     /* 0x000cff82 bit 4 */
      Bit8u disp_plane;           /* 0x000cff82 bit 5, 2-0 */
      Bit8u MIX_reg;              /* 0x000cff80 */
      Bit8u write_plane;          /* 0x000cff81 bit 3-0 */
      Bit8u read_plane;           /* 0x000cff81 bit 7-6 */
      Bit8u access_page_select;   /* 0x000cff83 bit 4 */
      Bit8u kanji_code_H;         /* 0x000cff94 */
      Bit8u kanji_code_L;         /* 0x000cff95 */
      Bit8u kanji_row_count;
      Bit32u kanji_offset;
      Boolean TVRAM_MD;           /* 0x05c8 bit 7 */
      Bit8u status_reg;           /* 0x000cff86 */
      Bit8u sub_status_reg;       /* 0xfda0 */
      } FMR_compat;

    struct {
      Bit8u addr;                 /* 0xfd90 */
      peldata data16_0[16];
      peldata data16_1[16];
      peldata data256_0[256];
      peldata data256_1[256];
      } pel;

    struct {
      Bit8u addr;                 /* 0x0458 */
      Bit8u mask[4];              /* 0x045a, 0x045b */
      } vram_ctrl;

    double  vsync_rate;
    double  hsync_rate;
    double  vsync_width;
    double  hsync_width;
    double  vdisp_top[2];
    double  vdisp_bottom[2];
    double  hdisp_top[2];
    double  hdisp_bottom[2];
    Boolean vsync_enable;
    Boolean vsync_trigered;
    Boolean field;
    Bit64u  vsync_trigered_time;
    layer_info linfo[2];
    Boolean  partial_updated;
    Boolean  all_updated;
    Boolean  dispmode_updated;
    unsigned x_dispsize;
    unsigned y_dispsize;
    unsigned x_tilesize;
    unsigned y_tilesize;
    Boolean  vga_tile_updated[BX_NUM_X_TILES][BX_NUM_Y_TILES];
    Bit8u video_memory[256 * 1024 * 2];
    Bit8u rgb[3 * 256];
    Bit32u tile[X_TILESIZE * Y_TILESIZE];
/* format
   33222222 22221111 111111
   10987654 32109876 54321098 76543210

   0000000l xxxxxxxx xxxxxxxx xxxxdddd   16 color (layer is l)
   0001000l xxxxxxxx xxxxxxxx dddddddd  256 color (layer is l)
   00100000 xxxxxxxx xGGGGGRR RRRBBBBB high color
   00110000 ???????? ???????? ???????? true color
   */
    } s;  // state information


  bx_devices_c *devices;

#if !BX_USE_VGA_SMF
  Bit32u read(Bit32u address, unsigned io_len);
  void   write(Bit32u address, Bit32u value, unsigned io_len, Boolean no_log);
#else
  void write(Bit32u address, Bit32u value, unsigned io_len, Boolean no_log);
#endif

  public:
  static void   timer_handler(void *);
  BX_VGA_SMF void   timer(void);
  static void   vsync_timer_handler(void *);
  BX_VGA_SMF void   vsync_timer(void);
  BX_VGA_SMF void   calculate_hvsync(void);
  private:
  BX_VGA_SMF void   update(void);
  BX_VGA_SMF void   dump_status(void);
  BX_VGA_SMF void   mem_update(Bit32u addr, Bit8u value);
  BX_VGA_SMF void   video_mode_test(void);
  BX_VGA_SMF int current_peltype(void);
  BX_VGA_SMF void pel_update(int type,
			     int number,
			     Boolean do_red,
			     Bit8u red,
			     Boolean do_green,
			     Bit8u green,
			     Boolean do_blue,
			     Bit8u blue);
  BX_VGA_SMF peldata *pel_address(int type,Bit8u number);
  BX_VGA_SMF void sprite_update(void);
  };

extern bx_vga_c bx_vga;

  inline int
bx_vga_c::current_peltype(void)
{
  switch(BX_VGA_THIS s.misc_output.shifter[1]&0x30) {
  case 0x00:
    return(PEL_16_0);

  case 0x20:
    return(PEL_16_1);

  case 0x10:
  case 0x30:
    return(PEL_256_0);
  }
  return(0);
}

  inline peldata
*bx_vga_c::pel_address(int type,Bit8u number)
{
  switch(type) {
  case PEL_16_0:
    return(&(BX_VGA_THIS s.pel.data16_0[number&0xf]));

  case PEL_16_1:
    return(&(BX_VGA_THIS s.pel.data16_1[number&0xf]));

  case PEL_256_0:
    return(&(BX_VGA_THIS s.pel.data256_0[number]));

  case PEL_256_1:
    return(&(BX_VGA_THIS s.pel.data256_1[number]));

  default:
    bx_vga.panic("vga: pel_update illegal type %d.\n",type);
    return 0;
  }
}

  inline void
bx_vga_c::pel_update(int type,
		     int number,
		     Boolean do_red,
		     Bit8u red,
		     Boolean do_green,
		     Bit8u green,
		     Boolean do_blue,
		     Bit8u blue)
{
  int maxnum,i;
  peldata *p;

  switch(type) {
  case PEL_16_0:
  case PEL_16_1:
    maxnum=0xf;
    break;

  case PEL_256_0:
  case PEL_256_1:
    maxnum=0xff;
    break;

  default:
    bx_vga.panic("vga: pel_update illegal type %d.\n",type);
    return;
  }

  if(number<0) {
    for(i=0;i<=maxnum;i++) {
      pel_update(type,i,do_red,red,do_green,green,do_blue,blue);
    }
  }

  p=pel_address(type,number);
  if(do_red) {
    p->red=red;
  }
  if(do_green) {
    p->green=green;
  }
  if(do_blue) {
    p->blue=blue;
  }

  switch(type) {
  case PEL_16_0:
  case PEL_16_1:
    if(bx_gui.palette_change(type,
			     number & maxnum,
			     p->red>>4,
			     p->green>>4,
			     p->blue>>4,
			     4)) {
      BX_VGA_THIS s.all_updated=1;
    }
    return;

  case PEL_256_0:
  case PEL_256_1:
    if(bx_gui.palette_change(type,
			     number & maxnum,
			     p->red,
			     p->green,
			     p->blue,
			     8)) {
      BX_VGA_THIS s.all_updated=1;
    }
    return;
  }
}
